/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef MvQFolderModel_H
#define MvQFolderModel_H

#include <QAbstractItemModel>
#include <QAbstractListModel>
#include <QHash>
#include <QFont>
#include <QFontMetrics>
#include <QSortFilterProxyModel>

#include "IconObject.h"
#include "FolderObserver.h"

class QMimeData;

class Folder;
class IconClass;
class IconObject;
class MvQFolderItemProperty;
class FolderSearchData;


class PositionHint
{
public:
    PositionHint() :
        className(""),
        x(0),
        y(0) {}
    void clear()
    {
        className = "";
        x         = 0;
        y         = 0;
    }
    string className;
    int x;
    int y;
};

class MvQFolderModel : public QAbstractItemModel, public FolderObserver, public IconObserver
{
    Q_OBJECT

public:
    MvQFolderModel(QObject* parent = 0);
    MvQFolderModel(int, QObject* parent = 0);
    ~MvQFolderModel();

    enum CustomItemRole
    {
        ClassFilterRole = Qt::UserRole + 1,
        EditorRole      = Qt::UserRole + 2,
        PositionRole    = Qt::UserRole + 3,
        FontMetricsRole = Qt::UserRole + 4
    };

    int columnCount(const QModelIndex&) const;
    int rowCount(const QModelIndex& parent = QModelIndex()) const;

    QVariant data(const QModelIndex&, int role = Qt::DisplayRole) const;
    bool setData(const QModelIndex&, const QVariant&, int);
    QVariant headerData(int, Qt::Orientation, int role = Qt::DisplayRole) const;

    QModelIndex index(int, int, const QModelIndex& parent = QModelIndex()) const;
    QModelIndex parent(const QModelIndex&) const;
    Qt::ItemFlags flags(const QModelIndex&) const;

    void folderIsAboutToChange();
    Folder* folder() { return folder_; }
    bool setFolder(Folder*, int iconSize = -1);
    bool setFolder(QString, int iconSize = -1);
    bool setFolder(const QModelIndex&, int iconSize = -1);
    bool setFolderToParent(int iconSize = -1);

    IconObject* objectFromIndex(const QModelIndex&) const;
    QModelIndex indexFromObject(IconObject*);
    QString fullName();
    QString fullName(const QModelIndex&);
    bool isFolder(const QModelIndex&);
    QModelIndex lastArrived();

    void createNewFolder();
    void saveFolderInfo();

    //Observer methods

    void arrived(IconObject*);
    void gone(IconObject*);
    void position(IconObject*, int, int);
    void renamed(IconObject*, const string&);

    void waiting(IconObject*);
    void error(IconObject*);
    void modified(IconObject*);
    void ready(IconObject*);
    void opened(IconObject*);
    void closed(IconObject*);
    void highlight(IconObject*);
    void iconChanged(IconObject*);

    void setIconSize(int);
    int iconSize() const;
    MvQFolderItemProperty* itemProp() const { return itemProp_; }

    void setAcceptedClasses(const vector<string>&);
    bool isAccepted(const IconClass&) const;
    void setPositionHint(const string&, int, int);

    void setSearchData(FolderSearchData*);
    void removeSearchData();

signals:
    void iconSizeChanged();
    void folderChanged(Folder*);
    void objectArrived(const QModelIndex&);
    void objectRenamed(const QModelIndex&, QString);

protected:
    void loadItems();
    bool isDataSet() const;
    QString formatFileSize(qint64) const;
    QString formatFileDate(time_t) const;
    bool isAccepted(IconObject*) const;
    void objectChanged(IconObject*);
    void updateSearchData(bool doReset);
    bool hasSearchData() const;

    Folder* folder_;
    QList<IconObject*> items_;
    bool initialScanIsOn_;
    vector<string> classes_;
    IconObject* lastArrived_;
    MvQFolderItemProperty* itemProp_;
    PositionHint posHint_;

    static QHash<IconObject::IconStatus, QColor> statusColour_;
    static QColor fadedColour_;

    mutable FolderSearchData* searchData_;

private:
    void init();
};


class MvQFolderFilterModel : public QSortFilterProxyModel
{
public:
    MvQFolderFilterModel(QObject* parent = 0);
    void setSourceModel(QAbstractItemModel*);
    bool filterAcceptsRow(int, const QModelIndex&) const;
    bool lessThan(const QModelIndex& left,
                  const QModelIndex& right) const;

    //QVariant data(const QModelIndex& index, int role ) const;
    //bool setFilter(QString,QString);

protected:
    MvQFolderModel* folderModel_;
    std::string name_;
    std::string type_;
    QList<IconObject*> matchLst_;
};

#endif
